// ==UserScript==
// @name         SeaArt 自動削除
// @namespace    http://tampermonkey.net/
// @version      6.1
// @description  自動削除ON時点を基準に、今日の古いタスク＋昨日以前を上から順に削除。日付が無い超古いタスクは、ON前にユーザーがタスク情報を開いた基準タスクより上だけ削除。削除対象が無くなってから5秒以上経過したときだけ自動停止。
// @match        https://www.seaart.ai/*
// @grant        none
// @run-at       document-end
// ==/UserScript==

(function () {
    'use strict';

    console.log('[TM] SeaArt 自動削除 v6.1 開始');

    // --- 状態 ---
    let autoDelete = false;

    // 今日のタスクを何時間以上前まで消すか
    let thresholdHours =
        parseFloat(localStorage.getItem('seaart_autoDelete_hours')) || 1.0;

    // 自動削除ON時点
    let baseNow = null;      // Date
    let baseNowTs = null;    // ms
    let cutoffTs = null;     // baseNowTs - thresholdHours 時間
    let todayStartTs = null; // 当日の 00:00:00

    let loopTimer = null;
    let isProcessing = false;

    // 削除対象が見つからなかったラウンド管理（5秒ディレイ用）
    let noDeleteFlag = false;
    let firstNoDeleteTime = 0;

    // 手動でタスク情報を開いたタスク
    let lastInfoTaskBar = null; // 直近でタスク情報ボタンを押したタスク
    let boundaryTaskBar = null; // 自動削除ON時点で採用する基準タスク

    // --- UI パネル ---
    const panel = document.createElement('div');
    panel.style.position = 'fixed';
    panel.style.right = '20px';
    panel.style.bottom = '20px';
    panel.style.zIndex = '999999';
    panel.style.display = 'flex';
    panel.style.gap = '6px';
    panel.style.alignItems = 'center';
    panel.style.fontSize = '12px';
    panel.style.background = 'rgba(0,0,0,0.4)';
    panel.style.padding = '6px 8px';
    panel.style.borderRadius = '8px';
    panel.style.backdropFilter = 'blur(4px)';
    document.body.appendChild(panel);

    const toggleBtn = document.createElement('button');
    toggleBtn.style.padding = '6px 10px';
    toggleBtn.style.border = 'none';
    toggleBtn.style.borderRadius = '6px';
    toggleBtn.style.cursor = 'pointer';
    toggleBtn.style.fontSize = '12px';

    function updateToggleBtn() {
        toggleBtn.textContent = '自動削除: ' + (autoDelete ? 'ON' : 'OFF');
        toggleBtn.style.background = autoDelete ? '#4CAF50' : '#f44336';
        toggleBtn.style.color = '#fff';
    }
    updateToggleBtn();

    const label = document.createElement('span');
    label.textContent = '作成';

    const input = document.createElement('input');
    input.type = 'number';
    input.min = '0';
    input.step = '0.1';
    input.style.width = '50px';
    input.style.fontSize = '12px';
    input.style.padding = '2px 4px';
    input.value = thresholdHours.toString();

    const label2 = document.createElement('span');
    label2.textContent = '時間以上前を削除';

    const boundaryLabel = document.createElement('span');
    boundaryLabel.style.marginLeft = '6px';
    boundaryLabel.textContent = '基準: なし';

    function updateBoundaryLabel() {
        boundaryLabel.textContent = boundaryTaskBar ? '基準: 設定済' : '基準: なし';
    }

    panel.appendChild(toggleBtn);
    panel.appendChild(label);
    panel.appendChild(input);
    panel.appendChild(label2);
    panel.appendChild(boundaryLabel);

    // 時間入力変更
    input.addEventListener('change', () => {
        const v = parseFloat(input.value);
        thresholdHours = isNaN(v) || v < 0 ? 0 : v;
        localStorage.setItem('seaart_autoDelete_hours', String(thresholdHours));
        console.log('[TM] 削除対象時間を更新:', thresholdHours, '時間');
    });

    // --- 手動でタスク情報を開いたタスクを覚える ---
    // 自動削除がオフの時にユーザーが押したタスク情報ボタンを基準候補にする
    document.addEventListener('click', (ev) => {
        if (autoDelete) return; // 自動削除中のクリックは無視

        const infoBtn = ev.target.closest('button[data-event="generate_pop_click_task_info"]');
        if (infoBtn) {
            const bar = infoBtn.closest('.pop-btns.flex');
            if (bar) {
                lastInfoTaskBar = bar;
                console.log('[TM] 基準候補タスクを更新しました（タスク情報ボタンから）');
            }
        }
    }, true);

    // --- ON / OFF ---
    toggleBtn.addEventListener('click', () => {
        autoDelete = !autoDelete;
        updateToggleBtn();

        if (autoDelete) {
            baseNow = new Date();
            baseNowTs = baseNow.getTime();
            cutoffTs = baseNowTs - thresholdHours * 60 * 60 * 1000;
            todayStartTs = new Date(
                baseNow.getFullYear(),
                baseNow.getMonth(),
                baseNow.getDate(),
                0, 0, 0, 0
            ).getTime();

            noDeleteFlag = false;
            firstNoDeleteTime = 0;

            // このタイミングで基準タスクを確定
            boundaryTaskBar = lastInfoTaskBar || null;
            updateBoundaryLabel();
            if (boundaryTaskBar) {
                console.log('[TM] 基準タスクを設定しました（タスク情報で選択）');
                try {
                    document.querySelectorAll('.seaart-boundary-highlight').forEach(el => {
                        el.classList.remove('seaart-boundary-highlight');
                        el.style.outline = '';
                    });
                    boundaryTaskBar.classList.add('seaart-boundary-highlight');
                    boundaryTaskBar.style.outline = '2px solid #ffd54f';
                } catch (e) {
                    console.log('[TM] 基準ハイライト設定エラー', e);
                }
            } else {
                console.log('[TM] 基準タスクは未設定です（基準なしモード）');
            }

            console.log('[TM] 自動削除 ON baseNow=', baseNow.toString(),
                        ' cutoffTs=', cutoffTs,
                        ' todayStartTs=', todayStartTs);

            startLoop();
        } else {
            stopLoop();
        }
    });

    // --- ヘルパー ---

    // 文字列から日時を取る
    // 2025/11/26 12:45:38 または 11:45:38 のどちらにも対応
    function parseCreateDateTime(text) {
        const t = text.trim();

        // 日付＋時間
        let m = t.match(/(\d{4})[\/\-](\d{1,2})[\/\-](\d{1,2})\s+(\d{1,2}):(\d{2})(?::(\d{2}))?/);
        if (m) {
            const year = parseInt(m[1], 10);
            const month = parseInt(m[2], 10) - 1;
            const day = parseInt(m[3], 10);
            const h = parseInt(m[4], 10);
            const mi = parseInt(m[5], 10);
            const s = m[6] ? parseInt(m[6], 10) : 0;
            return new Date(year, month, day, h, mi, s, 0);
        }

        // 時刻だけ（保険）
        m = t.match(/(\d{1,2}):(\d{2})(?::(\d{2}))?/);
        if (m) {
            const base = baseNow || new Date();
            const h = parseInt(m[1], 10);
            const mi = parseInt(m[2], 10);
            const s = m[3] ? parseInt(m[3], 10) : 0;
            return new Date(
                base.getFullYear(),
                base.getMonth(),
                base.getDate(),
                h, mi, s, 0
            );
        }

        return null;
    }

    // bar が boundaryTaskBar より上にあるかどうか
    function isAboveBoundary(bar) {
        if (!boundaryTaskBar) return false;
        const bars = Array.from(document.querySelectorAll('.pop-btns.flex'));
        const bi = bars.indexOf(bar);
        const boundIndex = bars.indexOf(boundaryTaskBar);
        if (bi === -1 || boundIndex === -1) return false;
        return bi < boundIndex; // DOM上で index が小さいほど上
    }

    function clickConfirmButtonWithPolling(onDone) {
        let waited = 0;
        const interval = 200;
        const maxWait = 3000;

        const timer = setInterval(() => {
            waited += interval;

            const footers = document.querySelectorAll('.easy-select-model-confirm-modal-footer');
            if (footers.length > 0) {
                const confirmBtn =
                    footers[footers.length - 1].querySelector('.modal-footer-btn-confirm');
                if (confirmBtn) {
                    console.log('[TM] 確認ボタンをクリック');
                    confirmBtn.click();
                    clearInterval(timer);
                    onDone && onDone(true);
                    return;
                }
            }

            if (waited >= maxWait) {
                console.log('[TM] 確認ボタン タイムアウト');
                clearInterval(timer);
                onDone && onDone(false);
            }
        }, interval);
    }

    function closeTaskInfoDialog() {
        const closeBtn = document.querySelector(
            '.easy-task-info-popup-header .modal-header-close'
        );
        if (closeBtn) {
            closeBtn.click();
        }
    }

    // --- メインループ ---
    function startLoop() {
        if (loopTimer) clearInterval(loopTimer);
        loopTimer = setInterval(processOneRound, 2000);
    }

    function stopLoop() {
        if (loopTimer) {
            clearInterval(loopTimer);
            loopTimer = null;
        }
    }

    // 1ラウンドで上から順に走査し、最初に削除条件を満たすタスクを1件だけ削除
    function processOneRound() {
        if (!autoDelete || isProcessing || cutoffTs == null || baseNowTs == null) return;
        isProcessing = true;

        const taskBars = Array.from(document.querySelectorAll('.pop-btns.flex'));
        if (taskBars.length === 0) {
            console.log('[TM] タスク無し → 即自動削除OFF');
            autoDelete = false;
            updateToggleBtn();
            stopLoop();
            isProcessing = false;
            return;
        }

        let index = 0;
        let deletedInThisRound = false;

        const scanNext = () => {
            if (index >= taskBars.length) {
                // このラウンドでは1件も削除していない
                if (!deletedInThisRound) {
                    handleNoDeleteRound();
                }
                isProcessing = false;
                return;
            }

            const bar = taskBars[index++];
            const infoBtn = bar.querySelector(
                'button[data-event="generate_pop_click_task_info"]'
            );
            const deleteBtn = bar.querySelector(
                'button[data-event="generate_pop_click_delete_all"]'
            );

            if (!infoBtn || !deleteBtn) {
                console.log('[TM] index', index - 1, ' info/delete ボタン無し → 次へ');
                scanNext();
                return;
            }

            console.log('[TM] タスク index=', index - 1, ' をチェック');
            infoBtn.click();

            setTimeout(() => {
                const box = document.querySelector('.easy-tooltip-wraper-params-box');
                if (!box) {
                    console.log('[TM] パラメータボックス無し → 次へ');
                    closeTaskInfoDialog();
                    scanNext();
                    return;
                }

                const items = box.querySelectorAll('.easy-tooltip-params-item');
                let rawText = null;

                items.forEach(item => {
                    const title = item.querySelector('.easy-tooltip-params-item-title');
                    const content = item.querySelector('.easy-tooltip-params-item-content');
                    if (title && content && title.textContent.includes('作成日時')) {
                        rawText = (content.textContent || '').trim();
                    }
                });

                let shouldDelete = false;

                if (!rawText) {
                    console.log('[TM] 作成日時テキスト無し');
                    // 日付が無い → 基準より上なら削除
                    if (boundaryTaskBar && isAboveBoundary(bar)) {
                        console.log('[TM] 日付無し ＋ 基準より上 → 強制削除対象');
                        shouldDelete = true;
                    } else {
                        console.log('[TM] 日付無し ＋ 基準未満または基準なし → 削除対象外');
                        shouldDelete = false;
                    }
                } else {
                    const created = parseCreateDateTime(rawText);
                    if (!created) {
                        console.log('[TM] 作成日時解析失敗 raw=', rawText);
                        if (boundaryTaskBar && isAboveBoundary(bar)) {
                            console.log('[TM] 解析失敗 ＋ 基準より上 → 強制削除対象');
                            shouldDelete = true;
                        } else {
                            console.log('[TM] 解析失敗 ＋ 基準未満または基準なし → 削除対象外');
                            shouldDelete = false;
                        }
                    } else {
                        const createdTs = created.getTime();
                        console.log('[TM] 作成日時 =', created.toString(), ' (ts=', createdTs, ')');

                        if (createdTs >= baseNowTs) {
                            console.log('[TM] ON以降作成 → 削除対象外');
                            shouldDelete = false;
                        } else if (createdTs >= todayStartTs) {
                            shouldDelete = createdTs <= cutoffTs;
                            console.log('[TM] 今日のタスク → cutoff 判定 shouldDelete=', shouldDelete);
                        } else {
                            shouldDelete = true;
                            console.log('[TM] 昨日以前のタスク → 無条件削除');
                        }
                    }
                }

                closeTaskInfoDialog();

                if (!shouldDelete) {
                    scanNext();
                    return;
                }

                // 削除対象が見つかった
                deletedInThisRound = true;
                noDeleteFlag = false;
                firstNoDeleteTime = 0;

                setTimeout(() => {
                    console.log('[TM] すべて削除ボタンをクリック');
                    deleteBtn.click();

                    clickConfirmButtonWithPolling(() => {
                        console.log('[TM] 削除完了 index=', index - 1);
                        // このラウンドは1件だけ削除して終了
                        isProcessing = false;
                    });
                }, 300);
            }, 400);
        };

        scanNext();
    }

    // このラウンドで1件も削除していない場合の処理（5秒遅延付き自動OFF）
    function handleNoDeleteRound() {
        const now = Date.now();

        if (!noDeleteFlag) {
            noDeleteFlag = true;
            firstNoDeleteTime = now;
            console.log('[TM] 削除対象無しを検出 1回目');
            return;
        }

        const elapsed = now - firstNoDeleteTime;
        console.log('[TM] 削除対象無し継続中 経過ms=', elapsed);

        if (elapsed >= 5000) {
            console.log('[TM] 5秒以上削除対象無し → 自動削除OFF');
            autoDelete = false;
            updateToggleBtn();
            stopLoop();
            noDeleteFlag = false;
            firstNoDeleteTime = 0;
        }
    }

})();
